<?php

use Models\Lockouts\Banning;
use Models\Lockouts\Error404;
use Models\Lockouts\LoginProtection;
use Models\Lockouts\Logs;

class TWS_IP_Lockouts
{

    protected static $instance = null;
    private $login_protection = null;
    private $ip = null;

    private static $IP_TYPE_SINGLE = 'single';
    private static $IP_TYPE_WILDCARD = 'wildcard';
    private static $IP_TYPE_MASK = 'mask';
    private static $IP_TYPE_CIDR = 'CIDR';
    private static $IP_TYPE_SECTION = 'section';
    private $_allowed_ips = array();

    private $models = array(
        "login_protection" => LoginProtection::class,
        "logs"             => Logs::class,
        "banning"          => Banning::class,
        "error404"         => Error404::class,
    );

    private function __construct()
    {
        // init models
        $this->ip = $this->get_client_ip();
        $this->init_models();

        //lockouts
        add_action("init", array($this, "check_blacklist"));

        if ($this->login_protection->enable_login_protection) {
            add_action('wp_login_failed', array($this, "login_failed"));
        }

        if ($this->error404->enable_404_detection) {
            add_action('template_redirect', array($this, "error404"), 1);
        }

    }

    public function check_blacklist_old()
    {
        // if ip in whitelist do nothing
        if (in_array($this->ip, $this->banning->whitelist)) {
            return false;
        }
        $block = false;
        if (in_array($this->ip, $this->banning->blacklist)) {
            $message = $this->banning->lockout_message;
            $block = true;

        } else {
            if ($this->login_protection->enable_login_protection) {
                $blacklist = $this->banning->getBlackListTemporary("login_failed");
                if (in_array($this->ip, $blacklist)) {
                    $message = $this->login_protection->lockout_message;
                    $block = true;
                }
            }
            if ($this->error404->enable_404_detection) {
                $blacklist = $this->banning->getBlackListTemporary("404error");
                if (in_array($this->ip, $blacklist)) {
                    $message = $this->error404->lockout_message;
                    $block = true;
                }
            }
        }

        // block ip
        if ($block === true) {
            require_once(TWS_DIR_VIEWS . '/Errors/error.php');
            die();
        }
    }

    private static function validateWildcardIpv4($ip, $pattern) {
        $pattern = rtrim($pattern, ".");
        $pattern .= str_repeat(".%", 3 - preg_match_all("/[.]/", $pattern));
        $pattern = str_replace(array('*'), '%', $pattern);
        $regExp = str_replace('%', '\d{1,3}', preg_quote(trim($pattern), '/'));
        return (boolean) preg_match("/^{$regExp}$/", $ip);
    }

    public function check_blacklist()
    {
        // if ip in whitelist do nothing
	    $is_banning_empty = empty($this->banning->whitelist);
        if (!$is_banning_empty && $this->check_ip($this->ip, $this->banning->whitelist)) {
            return false;
        }

        $block = $this->check_ip($this->ip, $this->banning->blacklist);
        $message = $this->banning->lockout_message;
        if(!$block){
            if ($this->login_protection->enable_login_protection) {
                $blacklist = $this->banning->getBlackListTemporary("login_failed");
                if (in_array($this->ip, $blacklist)) {
                    $message = $this->login_protection->lockout_message;
                    $block = true;
                }
            }
            if ($this->error404->enable_404_detection) {
                $blacklist = $this->banning->getBlackListTemporary("404error");
                if (in_array($this->ip, $blacklist)) {
                    $message = $this->error404->lockout_message;
                    $block = true;
                }
            }
        }

        // block ip
        if ($block === true) {
            require_once(TWS_DIR_VIEWS . '/Errors/error.php');
            die();
        }
    }

    public function login_failed($username)
    {
        $this->logs->username = $username;

        if (in_array($username, $this->login_protection->ban_usernames) && !in_array($this->ip, $this->banning->whitelist)) {
            $this->logs->type = "login_failed";
            $this->logs->addLog("locked");
            $this->banning->addBlackListTemporary($this->login_protection->lockout_time,"login_failed");
            return false;
        }
        $this->lockout("login_failed", $this->login_protection->lockout_time);
    }

    public function error404()
    {
        if (is_404()) {
            $uri = $_SERVER['REQUEST_URI'];

            if ( in_array( $uri, $this->error404->whitelist ) ) {
                return false;
            }
            $ext = pathinfo($uri, PATHINFO_EXTENSION);

            if($ext && in_array($ext,  $this->error404->ignore_file_types)){
                return false;
            }

            if($this->error404->exclusions == 1 && is_user_logged_in()){
                return false;
            }

            $this->lockout("404error", $this->error404->lockout_time);
        }
    }

    public static function get_instance()
    {
        if (null == self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    private function lockout($type, $time)
    {
        $this->logs->type = $type;
        $ip_log = $this->logs->getLogByIp();
        // check if expired
	    $is_ip_log_empty = empty($ip_log);
        if (!$is_ip_log_empty) {
            if (time() - strtotime($ip_log->date) > $this->login_protection->threshold_time_period) {
                $this->logs->updateLog($ip_log->id, "expired");
                $ip_log = null;
            }
        }
	    $is_ip_log_empty = empty($ip_log);
        if ($is_ip_log_empty) {
            $this->logs->addLog();
            $attempts = 1;
            $log_date = time();
        } else {
            $attempts = $ip_log->attempts + 1;
            $log_date = strtotime($ip_log->date);
            $this->logs->updateLog($ip_log->id, "active", $attempts, $this->logs->username);
        }


        if ($attempts >= $this->login_protection->threshold_attempts && time() - $log_date <= $this->login_protection->threshold_time_period) {
            $this->banning->ip = $this->ip;
            if (!in_array($this->ip, $this->banning->whitelist)) {
                $this->banning->addBlacklistTemporary($time, $type);
                $this->logs->updateLog($ip_log->id, "locked");
            }
        }
    }

    private function init_models()
    {
        foreach ($this->models as $prop_name => $model) {
            $this->$prop_name = new $model($this->ip);
        }
    }

    private function get_client_ip()
    {
        if (isset($_SERVER['HTTP_CLIENT_IP']))
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        else if (isset($_SERVER['HTTP_X_FORWARDED_FOR']))
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        else if (isset($_SERVER['HTTP_X_FORWARDED']))
            $ip = $_SERVER['HTTP_X_FORWARDED'];
        else if (isset($_SERVER['HTTP_FORWARDED_FOR']))
            $ip = $_SERVER['HTTP_FORWARDED_FOR'];
        else if (isset($_SERVER['HTTP_FORWARDED']))
            $ip = $_SERVER['HTTP_FORWARDED'];
        else if (isset($_SERVER['REMOTE_ADDR']))
            $ip = $_SERVER['REMOTE_ADDR'];
        else
            $ip = 'UNKNOWN';

        return $ip;
    }

    public function check_ip($ip, $allowed_ips = null) {
        $allowed_ips = $allowed_ips ? $allowed_ips : $this->_allowed_ips;
        foreach ($allowed_ips as $allowed_ip) {
            $type = $this->judge_ip_type($allowed_ip);
            $sub_rst = call_user_func(array($this, 'sub_checker_' . $type), $allowed_ip, $ip);

            if ($sub_rst) {
                return true;
            }
        }

        return false;
    }

    private function judge_ip_type($ip) {
        if (strpos($ip, '*')) {
            return self::$IP_TYPE_WILDCARD;
        }

        if (strpos($ip, '/')) {
            $tmp = explode('/', $ip);
            if (strpos($tmp[1], '.')) {
                return self::$IP_TYPE_MASK;
            } else {
                return self::$IP_TYPE_CIDR;
            }
        }

        if (strpos($ip, '-')) {
            return self::$IP_TYPE_SECTION;
        }

        /*if (ip2long($ip)) {
            return self::$IP_TYPE_SINGLE;
        }*/

        return self::$IP_TYPE_SINGLE;
    }

    private function sub_checker_single($allowed_ip, $ip) {
        return (ip2long($allowed_ip) == ip2long($ip));
    }

    private function sub_checker_wildcard($allowed_ip, $ip) {
        $allowed_ip_arr = explode('.', $allowed_ip);
        $ip_arr = explode('.', $ip);
        for ($i = 0; $i < count($allowed_ip_arr); $i++) {
            if ($allowed_ip_arr[$i] == '*') {
                return true;
            } else {
                if (false == ($allowed_ip_arr[$i] == $ip_arr[$i])) {
                    return false;
                }
            }
        }
    }

    private function sub_checker_mask($allowed_ip, $ip) {
        list($allowed_ip_ip, $allowed_ip_mask) = explode('/', $allowed_ip);
        $begin = (ip2long($allowed_ip_ip) & ip2long($allowed_ip_mask)) + 1;
        $end = (ip2long($allowed_ip_ip) | (~ ip2long($allowed_ip_mask))) + 1;
        $ip = ip2long($ip);
        return ($ip >= $begin && $ip <= $end);
    }

    private function sub_checker_section($allowed_ip, $ip) {
        list($begin, $end) = explode('-', $allowed_ip);
        $begin = ip2long(trim($begin));
        $end = ip2long(trim($end));
        $ip = ip2long($ip);
        return ($ip >= $begin && $ip <= $end);
    }

    private function sub_checker_CIDR($CIDR, $IP) {
        list ($net, $mask) = explode('/', $CIDR);
        return ( ip2long($IP) & ~((1 << (32 - $mask)) - 1) ) == ip2long($net);
    }



}



