<?php

namespace Models;

use Helpers\TwsDiff;

class Scan extends BaseModel
{
    private $exclude_from_themes = ["twentytwenty", "twentynineteen", "twentyseventeen", "twentysixteen"];
    private $exclude_from_plugins = [];
    private $manual_files = [WP_CONTENT_DIR . '/index.php', WP_CONTENT_DIR . '/plugins/index.php', WP_CONTENT_DIR . '/plugins/hello.php', WP_CONTENT_DIR . '/themes/index.php'];


    public function getHashes($type = "core", $slug = null)
    {
        $exclude_components = $this->exclude_from_plugins + $this->exclude_from_themes;
        if (!is_null($slug) && in_array($slug, $exclude_components)) {
            return false;
        }

        $dir = ABSPATH;

        if ($type == "plugin") {
            $dir = WP_CONTENT_DIR . '/plugins/' . $slug;

        } else if ($type == "theme") {
            $dir = WP_CONTENT_DIR . '/themes/' . $slug;
        }

        $hashes = $this->getCoreFiles($dir, $type);

        return $hashes;
    }

    public function getCoreThemes()
    {
        $core_themes = [];
        foreach (wp_get_themes() as $theme) {
            if (in_array($theme->get_template(), $this->exclude_from_themes)) {
                $core_themes[] = $theme->get_template();
            }
        }

        return $core_themes;
    }


    public function getPlugins()
    {
        $plugins = array();
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $exclude_components = array_merge($this->exclude_from_plugins, $this->exclude_from_themes);

        foreach (get_plugins() as $slug => $plugin) {
            $base_slug = explode('/', $slug);
            $base_slug = array_shift($base_slug);
            if (!in_array($base_slug, $exclude_components)) {
                $plugins[$base_slug] = array(
                    "version" => $plugin['Version'],
                    "type"    => 'plugin',
                    "title"   => $plugin['Name']
                );
            }
        }

        return $plugins;
    }

    public function getThemes()
    {
        $exclude_components = array_merge($this->exclude_from_plugins, $this->exclude_from_themes);
        $themes = array();
        foreach (wp_get_themes() as $theme) {
            if (!in_array($theme->get_template(), $exclude_components)) {
                $themes[$theme->get_template()] = array(
                    "version" => $theme->Version,
                    "type"    => 'theme',
                    "title"   => $theme->Name,
                );
            }
        }

        return $themes;
    }

    public function replaceToOrigFile($file, $type = "core", $version = "", $slug = "")
    {
        $file = ABSPATH . '/' . $file;
        $original_file = $this->getOriginalSource($file, $type, $version, $slug);

        if ($type == "plugin") {
            chmod(WP_PLUGIN_DIR, 0777);
        } else if ($type = "theme") {
            chmod(get_theme_root(), 0777);
        }
        $success = file_put_contents($file, $original_file);

        return $success;
    }

    public function getDiff($file, $type, $version, $slug, $file_status)
    {
        $original_file = '';
        $current_file = '';
        $diff = '';

        if ($file_status != 'added_file') {
            $original_file = $this->getOriginalSource($file, $type, $version, $slug);
        }
        if ($file_status != 'missing_file') {
            $current_file = file_get_contents($file);
            $diff = $this->textDiff($original_file, $current_file);
        }

        return array(
            "diff"          => $diff,
            "original_file" => explode("\n", $original_file),
            "file"          => explode("\n", $current_file)
        );
    }

    private function getOriginalSource($file, $type = "core", $version = "", $slug = "")
    {
        global $wp_version;

        $source_file_url = '';
        if ($type == "core") {
            $relative_path = str_replace(ABSPATH, '', $file);
            if (DIRECTORY_SEPARATOR == '\\') {
                $relative_path = str_replace('\\', '', $relative_path);
            }
            $source_file_url = "http://core.svn.wordpress.org/tags/" . $wp_version . "/" . $relative_path;
        } else {
            $file_name = substr($file, strpos($file, $slug) + strlen($slug));

            if ($type == "plugin") {
                $source_file_url = "http://plugins.svn.wordpress.org/" . $slug . "/tags/" . $version . "/" . $file_name;
            } else if ($type == "theme") {
                $source_file_url = "http://themes.svn.wordpress.org/" . $slug . "/" . $version . "/" . $file_name;
            }
        }
        $content = $source_file_url ? file_get_contents($source_file_url) : '';

        return $content;
    }


    private function textDiff($left_string, $right_string)
    {
        $diff = TwsDiff::compareFiles($left_string, $right_string);

        return $diff;
    }


    private function getCoreFiles($dir_path = ABSPATH, $type = "core")
    {
        $uploads_dir = wp_upload_dir();
        $uploads_dir["basedir"];
        $excluded_dirs = array('.svn', '.git', '.log', '.tmp', '.listing', '.cache', '.bak', '.swp', '~', 'cache', basename($uploads_dir["basedir"]));

        $files = array();
        if (!is_dir($dir_path)) {
            return false;
        }
        if (file_exists($dir_path)) {
            if ($type == "core") {
                $files = $this->getWpCoreFiles($dir_path, $excluded_dirs);
            } else {
                $files = $this->getWpComponentsFiles($dir_path, $excluded_dirs);
            }
        }

        return $files;
    }

    private function getWpCoreFiles($dir_path, $excluded_dirs)
    {
        $core_plugin_files = array();
        $core_theme_files = array();
        foreach ($this->exclude_from_plugins as $plugin) {
            $core_plugin_files += $this->getWpFiles(WP_CONTENT_DIR . '/plugins/' . $plugin, $excluded_dirs);
        }
        foreach ($this->exclude_from_themes as $theme) {
            $core_theme_files += $this->getWpFiles(WP_CONTENT_DIR . '/themes/' . $theme, $excluded_dirs);
        }
        $excluded_dirs[] = basename(WP_CONTENT_DIR);

        $files = $this->getWpFiles($dir_path, $excluded_dirs);
        // add some files manually :(
        foreach ($this->manual_files as $file_path) {
            if (file_exists($file_path)) {
                $file_name = str_replace(ABSPATH, '', $file_path);
                $files[$file_name] = hash_file("md5", $file_path);
            }
        }

        return $files + $core_plugin_files + $core_theme_files;
    }

    private function getWpComponentsFiles($dir_path, $excluded_dirs)
    {
        return $this->getWpFiles($dir_path, $excluded_dirs);
    }

    private function getWpFiles($dir_path, $excluded_dirs, &$files = array())
    {
        if (!is_dir($dir_path)) {
            return $files;
        }
        $file_paths = new \DirectoryIterator($dir_path);
        foreach ($file_paths as $file) {
            if ($file->isDot() || ($file->isDir() && in_array($file->getFilename(), $excluded_dirs))) {
                continue;
            }
            if (!$file->isDir()) {
                $file_path = $file->getRealPath();
                $file_name = str_replace(ABSPATH, '', $file_path);
                $files[$file_name] = hash_file("md5", $file_path);

            } else {
                $this->getWpFiles($file->getRealPath(), $excluded_dirs, $files);
            }
        }

        return $files;
    }


}