<?php
namespace Tenweb_Manager {

    include_once TENWEB_INCLUDES_DIR . '/class-api.php';
    include_once TENWEB_INCLUDES_DIR . '/class-amazon.php';
    include_once TENWEB_INCLUDES_DIR . '/class-amazon-multipart-upload.php';

    class MigrationMultiPartUpload extends Migration
    {
        /**
         * @var string $file
         */
        private $file;

        /**
         *  default is 5 mb
         *
         * @var integer $chunk_size
         */
        private $chunk_size;
        /**
         * @var integer $offset
         */
        private $offset = 0;
        /**
         * @var Amazon $amazon
         */
        private $amazon = 0;
        /**
         * @var AmazonMultiPartUpload $multipart_upload
         */
        private $multipart_upload;

        /**
         * MigrationMultiPartUpload constructor.
         *
         * @param     $file
         */
        public function __construct($file)
        {
            parent::__construct();
            $this->file = $file;
            $this->chunk_size = $this->configs['TENWEB_MIGRATION_MULTIPART_UPLOAD_CHUNK_SIZE'] * 1024 * 1024;
            $this->setAmazon();
        }


        /**
         * @param        $file
         * @param string $run_type
         *
         * @return bool
         */
        public static function run($file, $run_type = 'run')
        {
            if ($run_type == 'run') {
                Helper::store_migration_log('multipart_upload_run_type_' . current_time('timestamp'), 'Run type is run.');
                $upload_object = new self($file);

                return $upload_object->upload($run_type);
            }

            if ($run_type == 'restart') {
                Helper::store_migration_log('multipart_upload_run_type_' . current_time('timestamp'), 'Run type is restart.');
                $upload_object = Migration::get_object_file_content();

                return $upload_object->upload($run_type);
            }

            return false;
        }

        /**
         * @param string $run_type
         *
         * @return bool
         */
        public function upload($run_type = 'run')
        {
            $file_path = false;
            if ($run_type == 'run') {
                $this->multipart_upload = new AmazonMultiPartUpload($this->amazon);
                if (!$this->multipart_upload->initiate()) {
                    Helper::store_migration_log('error_in_multipart_upload_initiate' . current_time('timestamp'), json_encode($this->multipart_upload->getErrors()));

                    return false;
                }
            }

            Helper::store_migration_log('start_multipart_upload_part part number: ' . $this->multipart_upload->part_number, 'Started multipart upload part. Upload id is: ' . $this->multipart_upload->upload_id);

            if ($this->uploadPart()) {
                $file_path = $this->multipart_upload->complete();
                if ($file_path === false) {
                    Helper::store_migration_log('error_in_multipart_upload_complete' . current_time('timestamp'), json_encode($this->multipart_upload->getErrors()));

                    return false;
                }

                Helper::store_migration_log('complete_multipart_upload' . current_time('timestamp'), 'Complete multipart upload part. Upload id is: ' . $this->multipart_upload->upload_id);

            }


            return $file_path;
        }

        private function uploadPart()
        {
            $content = fopen($this->file, 'r');
            if (!$content) {
                Helper::store_migration_log('error_in_multipart_upload' . current_time('timestamp'), 'Empty file content. File does not exists or can not open it');

                return false;
            }

            $mime_type = mime_content_type($this->file);
            $file_size = filesize($this->file);
            fseek($content, $this->offset);

            while ($this->offset < $file_size) {
                $this->checkForRestart();

                $file_chunk = fread($content, $this->chunk_size);
                if (!$this->multipart_upload->uploadPart($file_chunk, $mime_type)) {
                    fclose($content);
                    Helper::store_migration_log('error_in_multipart_upload_part_upload' . current_time('timestamp'), json_encode($this->multipart_upload->getErrors()));

                    return false;
                }

                $this->offset += $this->chunk_size;
                fseek($content, $this->offset);
            }

            Helper::store_migration_log('end_multipart_upload_part part number: ' . $this->multipart_upload->part_number, 'End multipart upload part. Upload id is: ' . $this->multipart_upload->upload_id);

            return true;
        }

        private function checkForRestart()
        {
            $max_exec_time_server = ini_get('max_execution_time');
            $start = get_site_transient(TENWEB_PREFIX . "_migration_start_time");
            $script_exec_time = microtime(true) - $start;

            if ($script_exec_time >= ((int)$max_exec_time_server - $this->configs['TENWEB_MIGRATION_EXEC_TIME_OFFSET'])) {
                $this->writeObjectFile();
                $this->restart();
            }
        }

        private function writeObjectFile()
        {
            Helper::store_migration_log('start_write_object_file_multipart_upload_' . current_time('timestamp'), 'Starting write object file in multipart upload.');
            $this->wpdb = null;
            $content = serialize($this);
            file_put_contents($this->archive_dir . '/content_object.txt', $content);
            Helper::store_migration_log('end_write_object_file_multipart_upload_' . current_time('timestamp'), 'End write object file in multipart upload.');
        }

        private function setAmazon()
        {
            // get temp credentials
            $api = Api::get_instance();
            $aws_credentials = $api->get_amazon_tokens_for_migration(\TenwebServices::get_domain_id());
            if ($aws_credentials != null) {
                //can pass it as param, will change it
                $this->amazon = new Amazon(
                    $aws_credentials['credentials']['AccessKeyId'],
                    $aws_credentials['credentials']['SecretAccessKey'],
                    $aws_credentials['credentials']['SessionToken'],
                    $aws_credentials['folder'] . '/' . basename($this->file),
                    $aws_credentials['bucket'],
                    $aws_credentials['region']
                );
            } else {
                Helper::store_migration_log('migration_archive_uploading_to_s3', 'Error while getting credentials.');
            }
        }
    }
}
