<?php

namespace TmdHelper\App\Cli;

use TmdHelper\App\Exception\CliRestException;
use WP_CLI;
use WP_REST_Request;
use TmdHelper\App\Service\UserService;

class TmdCliRestApi
{

    /**
     * Do REST API internal request
     *
     * ## OPTIONS
     *
     * [--request-data=<jsonData>]
     * :Json data like: {"method":"GET","route":"settings","parameters":[]}
     *
     * @when after_wp_load
     */
    public function request(array $arguments, array $assocAgs)
    {
        /*
         * request-data structure example:
         * ['method' => 'GET', 'route' => 'users/1', 'parameters' => []]
         */

        /*
         * response data example:
         * ['status' => 'success|error', 'data' => [], 'message' => 'Some error message',  'code' => 200]
         */

        try {
            self::validateRequestData($assocAgs);
            $user = UserService::getFirstAdminUser();
            wp_set_current_user($user->ID); // it need for auth
            $requestData = json_decode($assocAgs['request-data'], true);
            $route = '/wp/v2/' . trim($requestData['route'], '/');
            $method = $requestData['method'];
            $parameters = $requestData['parameters'];

            // construct request
            $request = new WP_REST_Request($method, $route);
            if (in_array($method, array('POST', 'PUT'), true)) {
                $request->set_body_params($parameters);
            } else {
                foreach ($parameters as $key => $value) {
                    $request->set_param($key, $value);
                }
            }

            // call
            $response = rest_do_request($request);
            if ($response->is_error()) {
                $errorData = $response->get_data();
                self::returnResponse('error', [], $errorData['message'], 400);
            } else {
                self::returnResponse('success', $response->data);
            }
        } catch (\Exception $exception) {
            self::returnResponse('error', [], $exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * @param $status
     * @param $data
     * @param string $message
     * @param int $code
     */
    private static function returnResponse($status, $data, $message = '', $code = 200)
    {
        $response = [
            'status'    => $status,
            'data'      => $data,
            'message'   => $message,
            'code'      => $code
        ];

        WP_CLI::line(json_encode($response));
    }

    /**
     * @param $data
     * @throws CliRestException
     */
    private static function validateRequestData($data)
    {
        if (!$data['request-data'] || empty($data['request-data'])) {
            throw new CliRestException("'request-data' parameter missed or empty!", 404);
        }
        $data = json_decode($data['request-data'], true);
        if (!$data['method'] || empty($data['method'])) {
            throw new CliRestException("'method' parameter missed or empty!", 404);
        }
        if (!$data['route'] || empty($data['method'])) {
            throw new CliRestException("'route' parameter missed or empty!", 404);
        }
        if (!isset($data['parameters'])) {
            throw new CliRestException("'parameters' parameter missed or empty!", 404);
        }
    }
}
