<?php

namespace TmdHelper\App\Api\Signon;

use TmdHelper\App\Service\UserService;
use WP_Error, WP_REST_Response;

/**
 * Class SignonManager
 * Generates links for authorized redirect by our api and parses this link
 */
class SignonManager
{
    const AUTH_KEY_OPTION_KEY = 'tmd_auth_key';
    private $wpdb;

    /**
     * SignonManager constructor.
     * @param $wbdbInstance
     */
    public function __construct($wbdbInstance)
    {
        $this->wpdb = $wbdbInstance;
    }

    /**
     * @param array $data
     */
    public function doLoginAndRedirect(array $data)
    {
        $page = !empty($data['_p']) ? $data['_p'] : '/index.php';
        if (empty($token = $data['_t'])) {
            wp_redirect("/wp-login.php?redirect_to=$page&reauth=1");
            exit;
        }
        if (!$this->isTokenValid($token)) {
            wp_redirect("/wp-login.php?redirect_to=$page&reauth=1");
            exit;
        }
        unset($data['_p']);
        unset($data['_t']);
        unset($data['rest_route']);
        $page .= '&' . http_build_query($data);
        $this->setToken();
        wp_set_auth_cookie(UserService::getFirstAdminUser()->ID, false);
        wp_redirect($page);
        exit;
    }

    /**
     * @param $bodyParams
     * @return WP_Error|WP_REST_Response
     * @throws \Exception
     */
    public function saveAuthKey($bodyParams)
    {
        if (empty($bodyParams['_token']) || empty($bodyParams['expired'])) {
            return new WP_Error(400, "There are no required fields for storing key");
        }
        if ($this->isExpired($bodyParams['expired'])) {
            return new WP_Error(403, "Token had been expired");
        }
        $token = serialize($bodyParams);
        return $this->setToken($token);
    }

    /**
     * @param $timestamp
     * @return bool
     * @throws \Exception
     */
    private function isExpired($timestamp)
    {
        $now = new \DateTime('now', new \DateTimeZone('UTC'));
        return $now->getTimestamp() > $timestamp;
    }

    /**
     * @param $token
     * @return bool
     * @throws \Exception
     */
    private function isTokenValid($token)
    {
        $optionKey = self::AUTH_KEY_OPTION_KEY;
        $storedTokens = $this->wpdb->get_results("SELECT option_value FROM {$this->wpdb->options} WHERE option_name = '$optionKey'");
        if (empty($storedTokens)) {
            return false;
        }
        $storedToken = unserialize((current($storedTokens))->option_value);
        if (empty($storedToken['_token']) || empty($storedToken['expired'])) {
            return false;
        }
        if ($this->isExpired($storedToken['expired'])) {
            return false;
        }
        $testToken = md5(json_encode($storedToken));
        return $testToken === $token;
    }

    /**
     * @param string $token
     * @return WP_Error|WP_REST_Response
     */
    private function setToken($token = '')
    {
        $sql = "INSERT INTO {$this->wpdb->prefix}options (option_name,option_value) VALUES (%s,%s) ON DUPLICATE KEY UPDATE option_value = %s";
        $sql = $this->wpdb->prepare($sql, self::AUTH_KEY_OPTION_KEY, $token, $token);
        if (!$this->wpdb->query($sql)) {
            return new WP_Error(500, "Unable to save auth token");
        }
        return new WP_REST_Response(['success' => true], 201);
    }

}