<?php

namespace TmdHelper\App\Api\Matomo;

/**
 * Class Matomo_Configurator Manager
 * Performs configuring of Matomo plugin
 *
 * Provides interface for configuring wp-piwik (wp-matomo)
 */
class MatomoConfiguratorManager
{
    /**
     * @var Reference/wpdb
     */
    private $wpdb;

    /**
     * remote address of Matomo Server
     * @var string
     */
    protected $matomoAddr;

    /**
     * Matomo auth_token
     * @var string
     */
    protected $matomoToken;
    /**
     * Matomo tracking mode.
     * @var string
     */
    protected $matomoTrackMode = 'default';
    /**
     * Proxy url of Matomo server
     * @var string
     */
    protected $matomoProxyUrl;
    /**
     * id of site in database of Matomo Server
     * @var integer
     */
    protected $matomoSiteId;
    /**
     * noscript tag content
     * @var string
     */
    protected $matomoNoscript;
    /**
     * tracking code inside script tag
     * @var string
     */
    protected $matomoTrackingCode;
    /**
     * Unix timestamp of updating tracking code
     * @var integer
     */
    protected $matomoTrackingCodeUpdatedAt;

    /**
     * contains values of option_name field in the table wp_options, option_value of which must have value 'disabled'
     * @var array
     */
    private $valueDisabledInInsertedFields = [
        'wp-piwik_global-dashboard_widget',
    ];

    /**
     * contains format for generating noscript tag
     */
    const NOSCRIPT_FORMAT = '<noscript><p><img src="%s/matomo.php?idsite=%d&rec=1" style="border:0;" alt="" /></p></noscript>';

    /**
     * contains format for generating tracking code tag
     */
    const TRACKING_CODE_FORMAT = '<!-- Matomo -->
<script type="text/javascript">
  var _paq = window._paq || [];
  /* tracker methods like "setCustomDimension" should be called before "trackPageView" */
  _paq.push([\'trackPageView\']);
  _paq.push([\'enableLinkTracking\']);
  (function() {
    var u="%s";
    _paq.push([\'setTrackerUrl\', u+\'matomo.php\']);
    _paq.push([\'setSiteId\', \'%d\']);
    var d=document, g=d.createElement(\'script\'), s=d.getElementsByTagName(\'script\')[0];
    g.type=\'text/javascript\'; g.async=true; g.defer=true; g.src=u+\'matomo.js\'; s.parentNode.insertBefore(g,s);
  })();
</script>
<!-- End Matomo Code -->';

    /**
     * Array with values of option_name field that must be inserted into table
     * @var array
     */
    private $fieldsMustBeInserted = [
        'option_name' => [
            'wp-piwik_global-stats_seo',
            'wp-piwik_global-stats_ecommerce',
            'wp-piwik_global-dashboard_widget',
            'wp-piwik_global-dashboard_ecommerce',
            'wp-piwik_global-dashboard_chart',
            'wp-piwik_global-dashboard_seo',
            'wp-piwik_global-toolbar',
            'wp-piwik_global-perpost_stats',
            'wp-piwik_global-piwik_shortcut',
            'wp-piwik_global-shortcodes',
            'wp-piwik_global-track_noscript',
            'wp-piwik_global-track_nojavascript',
            'wp-piwik_global-track_search',
            'wp-piwik_global-track_404',
            'wp-piwik_global-add_customvars_box',
            'wp-piwik_global-disable_cookies',
            'wp-piwik_global-limit_cookies',
            'wp-piwik_global-track_admin',
            'wp-piwik_global-track_across',
            'wp-piwik_global-track_across_alias',
            'wp-piwik_global-track_crossdomain_linking',
            'wp-piwik_global-track_feed',
            'wp-piwik_global-track_feed_addcampaign',
            'wp-piwik_global-disable_timelimit',
            'wp-piwik_global-disable_ssl_verify',
            'wp-piwik_global-disable_ssl_verify_host',
            'wp-piwik_global-dnsprefetch',
            'wp-piwik_global-track_datacfasync',
        ],
    ];

    /**
     * list with names and values which must be updated
     * @var array
     */
    private $fieldsMustBeUpdated = [
        'option_name'  => [
            'wp-piwik_global-piwik_url',
            'wp-piwik_global-piwik_token',
            'wp-piwik_global-track_mode',
            'wp-piwik_global-proxy_url',
            'wp-piwik-site_id',
            'wp-piwik-noscript_code',
            'wp-piwik-tracking_code',
            'wp-piwik-last_tracking_code_update',
        ],
        'option_value' => [

        ],
    ];

    /**
     * Matomo_Configurator constructor.
     *
     * @param Reference/wpdb $wbdbInstance
     */
    public function __construct($wbdbInstance)
    {
        $this->wpdb = $wbdbInstance;
    }

    /**
     * @return $this
     * @throws \Exception
     */
    public function run()
    {
        if (empty($this->matomoTrackingCode)) {
            throw new \Exception('Tracking code is not generated');
        }
        $this->addConfiguration()
             ->updateConfiguration();

        return $this;
    }

    /**
     * @return $this
     */
    public function generateCodes()
    {
        $this->createMatomoNoscript()
             ->createMatomoTrackingCode()
             ->setMatomoTrackingCodeUpdatedAt();

        return $this;
    }

    /**
     * updates matomo configuration in DB
     */
    public function updateConfiguration()
    {
        //update db
        for ($i = 0; $i < count($this->fieldsMustBeUpdated['option_name']); $i++) {
            $this->wpdb->update(
                $this->wpdb->options,
                [
                    'option_name'  => $this->fieldsMustBeUpdated['option_name'][$i],
                    'option_value' => $this->fieldsMustBeUpdated['option_value'][$i],
                ],
                ['option_name' => $this->fieldsMustBeUpdated['option_name'][$i]],
                [
                    '%s',    // value1
                    '%s'    // value2
                ],
                ['%s']
            );
        }

        return $this;
    }

    /**
     * Inserts new rows with matomo configuration into db
     */
    public function addConfiguration()
    {
        $query = "INSERT INTO {$this->wpdb->options} (option_name, option_value, autoload) VALUES ";
        $values = [];
        $place_holders = [];
        foreach ($this->fieldsMustBeInserted['option_name'] as $key => $optionName) {
            array_push($values, $optionName);
            $placeHolder = ["'%s'"];
            if (in_array($optionName, $this->valueDisabledInInsertedFields)) {
                array_push($values, 'disabled');
                $placeHolder[] = "'%s'";
            } else {
                array_push($values, 0);
                $placeHolder[] = "'%d'";
            }
            array_push($values, 'yes');
            $placeHolder[] = "'%s'";

            $placeHolders[] = "(" . implode(',', $placeHolder) . ")";

            $query .= implode(', ', $placeHolders);
            $this->wpdb->query($this->wpdb->prepare("$query ", $values));
        }

        //insert into db
        return $this;
    }

    private function setValueToUpdate($fieldName, $value)
    {
        $this->fieldsMustBeUpdated['option_value'][array_search(
            $fieldName,
            $this->fieldsMustBeUpdated['option_name']
        )] = $value;
    }

    /**
     * @param $matomoAddr
     * @return $this
     * @throws \Exception
     */
    public function setMatomoAddr($matomoAddr)
    {
        if (!$this->validateMatomoAddr($matomoAddr)) {
            throw new \Exception('matomoAddr must begin from http: or https:');
        }
        $this->matomoAddr = $matomoAddr;
        $this->matomoProxyUrl = explode(':', $matomoAddr)[1];
        $this->setValueToUpdate('wp-piwik_global-piwik_url', $this->matomoAddr);
        $this->setValueToUpdate('wp-piwik_global-proxy_url', $this->matomoProxyUrl);
        $this->setValueToUpdate('wp-piwik_global-track_mode', $this->matomoTrackMode);

        return $this;
    }

    /**
     * @param $matomoToken
     * @return $this
     * @throws \Exception
     */
    public function setMatomoToken($matomoToken)
    {
        if (empty($matomoToken)) {
            throw new \Exception('matomo auth token mast be set');
        }
        $this->matomoToken = $matomoToken;
        $this->setValueToUpdate('wp-piwik_global-piwik_token', $this->matomoToken);

        return $this;
    }

    /**
     * @param integer $matomoSiteId
     *
     * @return $this
     */
    public function setMatomoSiteId($matomoSiteId)
    {
        $this->matomoSiteId = (integer) $matomoSiteId;
        $this->setValueToUpdate('wp-piwik-site_id', $this->matomoSiteId);

        return $this;
    }

    /**
     * @return $this
     * @throws \Exception
     */
    private function createMatomoNoscript()
    {
        if (empty($this->matomoSiteId) || empty($this->matomoProxyUrl)) {
            throw new \Exception('matomoSiteId or matomoProxyUrl cannot be empty');
        }

        $this->matomoNoscript = sprintf(self::NOSCRIPT_FORMAT, $this->matomoProxyUrl, $this->matomoSiteId);
        $this->setValueToUpdate('wp-piwik-noscript_code', $this->matomoNoscript);

        return $this;
    }

    /**
     * @return $this
     * @throws \Exception
     */
    private function createMatomoTrackingCode()
    {
        if (empty($this->matomoSiteId) || empty($this->matomoProxyUrl)) {
            throw new \Exception('matomoSiteId or matomoProxyUrl cannot be empty');
        }

        $this->matomoTrackingCode = sprintf(self::TRACKING_CODE_FORMAT, $this->matomoProxyUrl, $this->matomoSiteId);
        $this->setValueToUpdate('wp-piwik-tracking_code', $this->matomoTrackingCode);

        return $this;
    }

    /**
     * @return $this
     * @throws \Exception
     */
    private function setMatomoTrackingCodeUpdatedAt()
    {
        $this->matomoTrackingCodeUpdatedAt = (new \DateTime('now'))->format('U');
        $this->setValueToUpdate('wp-piwik-last_tracking_code_update', $this->matomoTrackingCodeUpdatedAt);

        return $this;
    }

    /**
     * @param $matomoAddr
     *
     * @return bool
     */
    private function validateMatomoAddr($matomoAddr)
    {
        return preg_match('/^https?:/', $matomoAddr) === 1;
    }
}
